#if NUNIT
using System;
using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;

using gov.va.med.vbecs.UnitTests;

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	/// <summary>
	///
	/// </summary>
	[TestFixture]
    public class Parsers_AutoInstrumentHL7Parser : BaseTest 

	{
		private HL7Interface intParms;
		private const string RBC_COMPONENT_ORDER = "2^RED BLOOD CELLS^99VBC^^";
		private const string FFP_COMPONENT_ORDER = "3^FRESH FROZEN PLASMA^99VBC^^";
		private const string CRYO_COMPONENT_ORDER = "5^CRYOPRECIPITATE^99VBC^^";
		private const string PLT_COMPONENT_ORDER = "4^PLATELETS^99VBC^^";
		private const string OTHER_COMPONENT_ORDER = "6^OTHER^99VBC^^";
		private const string WB_COMPONENT_ORDER = "11^WHOLE BLOOD^99VBC^^";
		private const string ABO_RH_TEST_ORDER = "7^ABO//RH^99VBC^^";
		private const string ABS_TEST_ORDER = "8^ABS^99VBC^^";
		private const string DAT_TEST_ORDER = "9^DAT^99VBC^^";
		private const string TAS_TEST_ORDER = "1^TAS^99VBC^^";
		private const string TRW_TEST_ORDER = "10^TRW^99VBC^^";
		private const string KC_DIVISION = "589^VA HEARTLAND - WEST, VISN 15";
		private const string WI_DIVISION = "589A7^WICHITA VAMC";


	    [SetUp]
		public void SetUp()
		{
			intParms = new HL7Interface( InterfaceName.Auto_Instrument.ToString() );
		}

		[TearDown]
		public void TearDown()
		{}

		[Test]
		public void Test_ParseHL7Message_Pass_1()
		{
            Guid rbcGuid = DataCreator.GetRandomBloodProduct(false, Common.ComponentClass.RBC).BloodProductGuid;    //matches hl7 msg
            BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, rbcGuid, ABO.A, RH.Positive, this.TestDivisionCode, false);
		    string tstMessage = DataCreator.GetHL7ORUMessage_UnitABORh(tmpUnit.LabelUnitId);       //APOs for now..

            AutoInstrumentHL7Parser _autoInstrumentHl7Parser = new AutoInstrumentHL7Parser();
		    HL7ProtocolMessage ackMessage = _autoInstrumentHl7Parser.ParseHL7Message(intParms, tstMessage);

			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode( ackMessage.GetMessage() ), "Ack Check" );
		}
		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void Test_ParseHL7Message_Fail_Null_Interface()
		{
		    AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
		    tmpParser.ParseHL7Message(null, "Null Interface");
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void Test_ParseHL7Message_Fail_Null_Message()
		{
            AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
		    HL7Interface tmpI = new HL7Interface(Common.InterfaceName.Auto_Instrument.ToString());
            tmpParser.ParseHL7Message(tmpI, null);
        }

        [Test]
        [ExpectedException(typeof(ArgumentNullException))]
        public void Test_ParseHL7Message_Fail_Empty_Message()
        {
            AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
            HL7Interface tmpI = new HL7Interface(Common.InterfaceName.Auto_Instrument.ToString());
            tmpParser.ParseHL7Message(tmpI, string.Empty);
        }

        [Test]
        [ExpectedException(typeof(HL7Exception))]
        public void Test_ParseHL7Message_Fail_Invalid_InterfaceName()
        {
            AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
            HL7Interface tmpI = new HL7Interface(Common.InterfaceName.CPRS.ToString());
            tmpParser.ParseHL7Message(tmpI, "Bad Interface");
        }

        [Test]
        [ExpectedException(typeof(Hl7AckException))]
        public void Test_ParseHL7Message_Fail_BadMessage()
        {
            AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
            HL7Interface tmpI = new HL7Interface(Common.InterfaceName.Auto_Instrument.ToString());
            string tstMsg = DataCreator.GetHL7ORUMessage_UnitABORh("FakeUnitID").Replace("589", "XXX");
            tmpParser.ParseHL7Message(tmpI, tstMsg);
        }

        [Test]
        [ExpectedException(typeof(HL7Exception))]
        public void Test_ParseHL7Message_Fail_BadMesageType()
        {
            AutoInstrumentHL7Parser tmpParser = new AutoInstrumentHL7Parser();
            HL7Interface tmpI = new HL7Interface(Common.InterfaceName.Auto_Instrument.ToString());
            string tstMsg = DataCreator.GetHL7ORUMessage_UnitABORh("000000001");
            tmpParser.ParseHL7Message(tmpI, tstMsg.Replace("ORU^R01", "POO^420"));
        }
    
    }
}
#endif